#!/bin/sh

# Copyright (c) 2022, Caleb Connolly <caleb@connolly.tech>
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


# Toggle between tty1 and tty2, launching $KEYBOARD when on tty2
# THIS SCRIPT MUST BE RUN AS ROOT
# usage:
# togglevt.sh <state>
# where <state> is an optional arg to require that a counter be incremented before the action
# is performed. The default configuration will perform the switch when the power button has
# been pressed 3 times whilst the volume down button is being held.
# if no arguments are specified the switch will occur immediately.

[ "$(whoami)" != root ] && echo "This must be run as root" && exit 1

# shellcheck disable=SC1091
test -f /etc/conf.d/ttyescape.conf && . /etc/conf.d/ttyescape.conf

# default font, override this by setting it in /etc/conf.d/ttyescape.conf
FONT="${FONT:-/usr/share/consolefonts/ter-128n.psf.gz}"
# amount of times power must be pressed to trigger
PRESSCOUNT="${PRESSCOUNT:-3}"
TMPFILE="${TMPFILE:-/tmp/ttyescape.tmp}"
KEYBOARD="${KEYBOARD:-buffyboard}"

if [ ! -e /dev/uinput ]; then
	if ! modprobe -q uinput; then
		echo "uinput module not available, please enable it in your kernel"
	fi
fi

switchtty() {
	currentvt=$(cat /sys/devices/virtual/tty/tty0/active)

	if [ "$currentvt" = "tty2" ]; then # switch to tty1 with normal UI
		chvt 1
		killall "$KEYBOARD"
	else # Switch to tty2 with $KEYBOARD
		setfont "$FONT" -C /dev/tty2
		chvt 2
		# sometimes they keyboard can be running already, we shouldn't start it in that case
		[ "$(pgrep "$KEYBOARD")" ] || nohup "$KEYBOARD" -r "$(cat /sys/class/graphics/fbcon/rotate)" &
	fi
}

# If we receive a command that isn't start
# and we don't have the file used to count
# then we should do nothing
if [ -n "$1" ] && [ "$1" != "start" ] && [ ! -f "$TMPFILE" ]; then
	exit 0
fi

case "$1" in
	# No args means just DO IT
	"")
		switchtty
		;;
	# Start counting, this should
	# run when voldown is pressed
	"start")
		echo "0" > "$TMPFILE"
		;;
	# Run when voldown releases
	"reset")
		rm "$TMPFILE"
		;;
	# Run when power pressed while
	# voldown is pressed
	"inc")
		val="$(cat "$TMPFILE")"
		val=$((val+1))
		if [ $val -eq "$PRESSCOUNT" ]; then
			rm "$TMPFILE"
			switchtty
		else
			echo "$val" > "$TMPFILE"
		fi
		;;
	*)
esac
